#!/usr/bin/cmake -DOUTPUT_PATH=iso_language_codes.cpp -P
# generates C++ code for ISO-639-2 language codes
cmake_minimum_required(VERSION 3.19.0 FATAL_ERROR)

# get language file if not specified
if (NOT LANGUAGE_FILE)
    # default to path provided usually by iso-codecs package (https://salsa.debian.org/iso-codes-team/iso-codes)
    set(LANGUAGE_FILE "/usr/share/iso-codes/json/iso_639-2.json")
    # download the file from upstream repo if it is not installed locally
    if (NOT EXISTS "${LANGUAGE_FILE}")
        set(LANGUAGE_FILE "resources/iso_639-2.json")
        message(STATUS "Downloading ${LANGUAGE_FILE}")
        file(DOWNLOAD "https://salsa.debian.org/iso-codes-team/iso-codes/-/raw/main/data/iso_639-2.json" "${LANGUAGE_FILE}" SHOW_PROGRESS TLS_VERIFY ON)
    endif()
endif ()
if (NOT EXISTS "${LANGUAGE_FILE}")
    message(FATAL_ERROR "The file ${LANGUAGE_FILE} does not exist.")
endif()
if (NOT OUTPUT_PATH)
    message(FATAL_ERROR "No OUTPUT_PATH specified.")
endif()

set(OUTPUT "static const std::unordered_map<std::string, std::string> languageNames_iso_639_2_b = {\n")
file(READ "${LANGUAGE_FILE}" LANGUAGE_JSON)
string(JSON LANGUAGE_COUNT LENGTH "${LANGUAGE_JSON}" "639-2")
message(STATUS "Found ${LANGUAGE_COUNT} ISO-639-2 language codes")
math(EXPR LANGUAGE_COUNT "${LANGUAGE_COUNT} - 1")
foreach (LANGUAGE_INDEX RANGE "${LANGUAGE_COUNT}")
    string(JSON LANGUAGE_ENTRY GET "${LANGUAGE_JSON}" "639-2" "${LANGUAGE_INDEX}")
    string(JSON LANGUAGE_ALPHA_2 ERROR_VARIABLE ERROR GET "${LANGUAGE_ENTRY}" "alpha_2")
    string(JSON LANGUAGE_ALPHA_3 ERROR_VARIABLE ERROR GET "${LANGUAGE_ENTRY}" "alpha_3")
    string(JSON LANGUAGE_BIBLIOGRAPHIC ERROR_VARIABLE ERROR GET "${LANGUAGE_ENTRY}" "bibliographic")
    string(JSON LANGUAGE_NAME ERROR_VARIABLE ERROR GET "${LANGUAGE_ENTRY}" "name")
    if (NOT LANGUAGE_BIBLIOGRAPHIC)
        set(LANGUAGE_BIBLIOGRAPHIC "${LANGUAGE_ALPHA_3}")
    endif ()
    set(OUTPUT "${OUTPUT}    {\"${LANGUAGE_BIBLIOGRAPHIC}\", \"${LANGUAGE_NAME}\"},\n")
endforeach ()
set(OUTPUT "${OUTPUT}};\n")
file(WRITE "${OUTPUT_PATH}" "${OUTPUT}")
message(STATUS "Wrote language entries to ${OUTPUT_PATH}")
